from __future__ import division, print_function
import os
import numpy as np
from tensorflow.keras.models import load_model
from tensorflow.keras.preprocessing import image
import tensorflow as tf
from flask import Flask, request, render_template
from werkzeug.utils import secure_filename
import cv2
import skimage.exposure

# Initialize Flask app
app = Flask(__name__)

# Create uploads folder if not exists
UPLOAD_FOLDER = os.path.join(os.path.dirname(__file__), 'uploads')
if not os.path.exists(UPLOAD_FOLDER):
    os.makedirs(UPLOAD_FOLDER)

# ==============================
# Custom Generalized Mean Pool
# ==============================
gm_exp = tf.Variable(3., dtype=tf.float32)
def generalized_mean_pool_2d(X):
    pool = (tf.reduce_mean(tf.abs(X**(gm_exp)), axis=[1,2], keepdims=False) + 1.e-8)**(1./gm_exp)
    return pool

# ==============================
# Load Model
# ==============================
# Update this path according to your project
path_to_model = os.path.join(os.path.dirname(__file__), 'model', 'oral_cancer_model.h5')

print("🔹 Loading the model...")
model = load_model(path_to_model, compile=False)
model.compile(optimizer='adam', metrics=['accuracy'], loss='categorical_crossentropy')
print("✅ Model loaded successfully!")

# ==============================
# Class Labels
# ==============================
classes = {
    0: 'Normal',
    1: 'OSCC'
}

# ==============================
# Prediction Function
# ==============================
def model_predict(img_path, model):
    img = cv2.imread(img_path)
    image_duplicate = img.copy()
    hsv = cv2.cvtColor(image_duplicate, cv2.COLOR_BGR2HSV)
    h, s, v = cv2.split(hsv)
    h_new = (h + 90) % 180
    hsv_new = cv2.merge([h_new, s, v])
    bgr_new = cv2.cvtColor(hsv_new, cv2.COLOR_HSV2BGR)
    ave_color = cv2.mean(bgr_new)[0:3]
    color_img = np.full_like(image_duplicate, ave_color)
    blend = cv2.addWeighted(image_duplicate, 0.5, color_img, 0.5, 0.0)
    result = skimage.exposure.rescale_intensity(blend, in_range='image', out_range=(0, 255)).astype(np.uint8)

    img = cv2.resize(result, (299, 299), interpolation=cv2.INTER_CUBIC)
    x = image.img_to_array(img)
    x = np.expand_dims(x, axis=0)
    x = x / 255.0

    preds = model.predict(x)
    pred_class = np.argmax(preds, axis=1)[0]
    return classes[pred_class]

# ==============================
# Flask Routes
# ==============================
@app.route('/', methods=['GET'])
def index():
    return render_template('index.html')

@app.route('/predict', methods=['POST'])
def upload():
    if 'file' not in request.files:
        return "⚠️ No file uploaded!"

    f = request.files['file']
    if f.filename == '':
        return "⚠️ No file selected!"

    file_path = os.path.join(UPLOAD_FOLDER, secure_filename(f.filename))
    f.save(file_path)

    preds = model_predict(file_path, model)
    return render_template('result.html', prediction=preds)

# ==============================
# Main Entry
# ==============================
if __name__ == '__main__':
    app.run(debug=True)
